/*******************************************************************************
 * Copyright (c) 1998, 2022 IBM Corp. and others
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] http://openjdk.java.net/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0 WITH Classpath-exception-2.0 OR LicenseRef-GPL-2.0 WITH Assembly-exception
 *******************************************************************************/

/**
 * @file
 * @ingroup Port
 * @brief Port Library Error Codes
 *
 * When an error is reported by the operating system the port library must translate this OS specific error code to a
 * portable error code.  All portable error codes are negative numbers.  Not every module of the port library will have
 * error codes dedicated to it's own use, some will use the generic portable error code values.
 *
 * Errors reported by the OS may be recorded by calling the port library functions @ref omrerror.c::omrerror_set_last_error "omrerror_set_last_error()"
 * or @ref omrerror.c::omrerror_set_last_error_with_message "omrerror_set_last_error_with_message()"
 * or @ref omrerror.c::omrerror_set_last_error_with_message_format "omrerror_set_last_error_with_message_format()".
 * The mapping of the OS specific error code to a portable error code is the responsibility of the calling module.
 */
#ifndef omrporterror_h
#define omrporterror_h


/**
 * @name Generic Errors
 * Generic error codes for modules that do not have their own specific errors.  Where possible it is recommended that all
 * modules that return error codes have their own set
 *
 * @internal OMRPORT_ERROR* range from -1 to -49 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_BASE -1
#define OMRPORT_ERROR_OPFAILED OMRPORT_ERROR_BASE
#define OMRPORT_ERROR_EOF (OMRPORT_ERROR_BASE-1)
#define OMRPORT_ERROR_NOPERMISSION (OMRPORT_ERROR_BASE-2)
#define OMRPORT_ERROR_NOTFOUND (OMRPORT_ERROR_BASE-3)
#define OMRPORT_ERROR_NAMETOOLONG (OMRPORT_ERROR_BASE-4)
#define OMRPORT_ERROR_DISKFULL (OMRPORT_ERROR_BASE-5)
#define OMRPORT_ERROR_EXIST (OMRPORT_ERROR_BASE-6)
#define OMRPORT_ERROR_NOTEXIST (OMRPORT_ERROR_BASE-7)
#define OMRPORT_ERROR_SYSTEMFULL (OMRPORT_ERROR_BASE-8)
#define OMRPORT_ERROR_NOENT (OMRPORT_ERROR_BASE-9)
#define OMRPORT_ERROR_DIRECTORY (OMRPORT_ERROR_BASE-10)
#define OMRPORT_ERROR_NOTDIRECTORY (OMRPORT_ERROR_BASE-11)
#define OMRPORT_ERROR_LOOP (OMRPORT_ERROR_BASE-12)
#define OMRPORT_ERROR_BADF (OMRPORT_ERROR_BASE-13)
#define OMRPORT_ERROR_WOULDBLOCK (OMRPORT_ERROR_BASE-14)
#define OMRPORT_ERROR_INVALID (OMRPORT_ERROR_BASE-15)
#define OMRPORT_ERROR_FILE_LOCK_VIOLATION (OMRPORT_ERROR_BASE-16)
#define OMRPORT_ERROR_INVALID_HANDLE (OMRPORT_ERROR_BASE-17)
#define OMRPORT_ERROR_NOT_SUPPORTED_ON_THIS_PLATFORM (OMRPORT_ERROR_BASE-18)
#define OMRPORT_ERROR_INVALID_ARGUMENTS (OMRPORT_ERROR_BASE-19)
/** @} */

/**
 * @name Port Library startup failure code
 * Failures related to the initialization and startup of the port library.
 *
 * @internal OMRPORT_ERROR_STARTUP* range from -50 to -99 to avoid overlap
 * @{
 */
#define OMRPORT_ERROR_STARTUP_BASE -50
#define OMRPORT_ERROR_STARTUP_THREAD (OMRPORT_ERROR_STARTUP_BASE)
#define OMRPORT_ERROR_STARTUP_MEM (OMRPORT_ERROR_STARTUP_BASE -1)
#define OMRPORT_ERROR_STARTUP_TLS (OMRPORT_ERROR_STARTUP_BASE -2)
#define OMRPORT_ERROR_STARTUP_TLS_ALLOC (OMRPORT_ERROR_STARTUP_BASE -3)
#define OMRPORT_ERROR_STARTUP_TLS_MUTEX (OMRPORT_ERROR_STARTUP_BASE -4)
#define OMRPORT_ERROR_STARTUP_CPU (OMRPORT_ERROR_STARTUP_BASE -5)
#define OMRPORT_ERROR_STARTUP_VMEM (OMRPORT_ERROR_STARTUP_BASE -6)
#define OMRPORT_ERROR_STARTUP_FILE (OMRPORT_ERROR_STARTUP_BASE -7)
#define OMRPORT_ERROR_STARTUP_TTY (OMRPORT_ERROR_STARTUP_BASE -8)
#define OMRPORT_ERROR_STARTUP_TTY_HANDLE (OMRPORT_ERROR_STARTUP_BASE -9)
#define OMRPORT_ERROR_STARTUP_TTY_CONSOLE (OMRPORT_ERROR_STARTUP_BASE -10)
#define OMRPORT_ERROR_STARTUP_MMAP (OMRPORT_ERROR_STARTUP_BASE -11)
#define OMRPORT_ERROR_STARTUP_NLS (OMRPORT_ERROR_STARTUP_BASE -12)
#define OMRPORT_ERROR_STARTUP_TIME (OMRPORT_ERROR_STARTUP_BASE -13)
#define OMRPORT_ERROR_STARTUP_EXIT (OMRPORT_ERROR_STARTUP_BASE -14)
#define OMRPORT_ERROR_STARTUP_SYSINFO (OMRPORT_ERROR_STARTUP_BASE -15)
#define OMRPORT_ERROR_STARTUP_SL (OMRPORT_ERROR_STARTUP_BASE -16)
#define OMRPORT_ERROR_STARTUP_STR (OMRPORT_ERROR_STARTUP_BASE -17)
#define OMRPORT_ERROR_STARTUP_ERROR (OMRPORT_ERROR_STARTUP_BASE -18)
#define OMRPORT_ERROR_STARTUP_SIGNAL (OMRPORT_ERROR_STARTUP_BASE -19)
#define OMRPORT_ERROR_STARTUP_AIX_PROC_ATTR (OMRPORT_ERROR_STARTUP_BASE -20)
#define OMRPORT_ERROR_STARTUP_DUMP (OMRPORT_ERROR_STARTUP_BASE -21)
#define OMRPORT_ERROR_STARTUP_CUDA (OMRPORT_ERROR_STARTUP_BASE -22)
#define OMRPORT_ERROR_STARTUP_SHSEM (OMRPORT_ERROR_STARTUP_BASE -23)
#define OMRPORT_ERROR_STARTUP_SHMEM (OMRPORT_ERROR_STARTUP_BASE -24)
#define OMRPORT_ERROR_STARTUP_SHMEM_MOSERVICE (OMRPORT_ERROR_STARTUP_BASE -25)
#define OMRPORT_ERROR_STARTUP_SYSINFO_MONITOR_INIT (OMRPORT_ERROR_STARTUP_BASE -26)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS (OMRPORT_ERROR_STARTUP_BASE -27)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS1 OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS2 (OMRPORT_ERROR_STARTUP_BASE -28)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS3 (OMRPORT_ERROR_STARTUP_BASE -29)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS4 (OMRPORT_ERROR_STARTUP_BASE -30)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS5 (OMRPORT_ERROR_STARTUP_BASE -31)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS6 (OMRPORT_ERROR_STARTUP_BASE -32)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS7 (OMRPORT_ERROR_STARTUP_BASE -33)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS8 (OMRPORT_ERROR_STARTUP_BASE -34)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS9 (OMRPORT_ERROR_STARTUP_BASE -35)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS10 (OMRPORT_ERROR_STARTUP_BASE -36)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS11 (OMRPORT_ERROR_STARTUP_BASE -37)
#define OMRPORT_ERROR_STARTUP_SIGNAL_TOOLS12 (OMRPORT_ERROR_STARTUP_BASE -38)
#define OMRPORT_ERROR_STARTUP_SYSINFO_RUNNING_IN_CONTAINER (OMRPORT_ERROR_STARTUP_BASE -39)

/** @} */

/**
 * File operations system function call errno mappings
 * @internal
 * Errno for following system call functions:
 *  fstat
 * 	fstatfs (on Linux)
 * 	fstatvfs (on AIX)
 *
 * The errors are meant to be a 32 bit integer.
 * The most significant 16 bits will identify the failing function
 * The least significant 16 bits will identify the errno that occured while calling the function
 *
 */
#define OMRPORT_ERROR_FILE_FSTAT_ERROR (-(2<<16))
#define OMRPORT_ERROR_FILE_FSTATFS_ERROR (-(3<<16))
#define OMRPORT_ERROR_FILE_FSTATVFS_ERROR (-(3<<16))

/**
 * @name File Errors
 * Error codes for file operations.
 *
 * @internal OMRPORT_ERROR_FILE* range from -100 to -149 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_FILE_BASE -100
#define OMRPORT_ERROR_FILE_OPFAILED (OMRPORT_ERROR_FILE_BASE)
#define OMRPORT_ERROR_FILE_EOF (OMRPORT_ERROR_FILE_BASE-1)
#define OMRPORT_ERROR_FILE_NOPERMISSION (OMRPORT_ERROR_FILE_BASE-2)
#define OMRPORT_ERROR_FILE_NOTFOUND (OMRPORT_ERROR_FILE_BASE-3)
#define OMRPORT_ERROR_FILE_NAMETOOLONG (OMRPORT_ERROR_FILE_BASE-4)
#define OMRPORT_ERROR_FILE_DISKFULL (OMRPORT_ERROR_FILE_BASE-5)
#define OMRPORT_ERROR_FILE_EXIST (OMRPORT_ERROR_FILE_BASE-6)
#define OMRPORT_ERROR_FILE_SYSTEMFULL (OMRPORT_ERROR_FILE_BASE-7)
#define OMRPORT_ERROR_FILE_NOENT (OMRPORT_ERROR_FILE_BASE-8)
#define OMRPORT_ERROR_FILE_NOTDIR (OMRPORT_ERROR_FILE_BASE-9)
#define OMRPORT_ERROR_FILE_LOOP (OMRPORT_ERROR_FILE_BASE-10)
#define OMRPORT_ERROR_FILE_BADF (OMRPORT_ERROR_FILE_BASE-11)
#define OMRPORT_ERROR_FILE_WOULDBLOCK (OMRPORT_ERROR_FILE_BASE-12)
#define OMRPORT_ERROR_FILE_INVAL (OMRPORT_ERROR_FILE_BASE-13)
#define OMRPORT_ERROR_FILE_LOCK_NOREADWRITE (OMRPORT_ERROR_FILE_BASE-14)
#define OMRPORT_ERROR_FILE_LOCK_NOWAITNOWAIT (OMRPORT_ERROR_FILE_BASE-15)
#define OMRPORT_ERROR_FILE_LOCK_BADLOCK (OMRPORT_ERROR_FILE_BASE-16)
#define OMRPORT_ERROR_FILE_UNLOCK_BADUNLOCK (OMRPORT_ERROR_FILE_BASE-17)
#define OMRPORT_ERROR_FILE_INTERRUPTED (OMRPORT_ERROR_FILE_BASE-18)
#define OMRPORT_ERROR_FILE_ISDIR (OMRPORT_ERROR_FILE_BASE-19)
#define OMRPORT_ERROR_FILE_PROCESS_MAX_OPEN (OMRPORT_ERROR_FILE_BASE-20)
#define OMRPORT_ERROR_FILE_XIO (OMRPORT_ERROR_FILE_BASE-21)
#define OMRPORT_ERROR_FILE_ROFS (OMRPORT_ERROR_FILE_BASE-22)
#define OMRPORT_ERROR_FILE_LOCK_EDEADLK (OMRPORT_ERROR_FILE_BASE-23)
#define OMRPORT_ERROR_FILE_EAGAIN (OMRPORT_ERROR_FILE_BASE-24)
#define OMRPORT_ERROR_FILE_EFAULT (OMRPORT_ERROR_FILE_BASE-25)
#define OMRPORT_ERROR_FILE_EINTR (OMRPORT_ERROR_FILE_BASE-26)
#define OMRPORT_ERROR_FILE_IO (OMRPORT_ERROR_FILE_BASE-27)
#define OMRPORT_ERROR_FILE_OVERFLOW (OMRPORT_ERROR_FILE_BASE-28)
#define OMRPORT_ERROR_FILE_SPIPE (OMRPORT_ERROR_FILE_BASE-29)
#define OMRPORT_ERROR_FILE_FSTAT_FAILED (OMRPORT_ERROR_FILE_BASE-30)
#define OMRPORT_ERROR_FILE_OPERATION_ABORTED (OMRPORT_ERROR_FILE_BASE-31)
#define OMRPORT_ERROR_FILE_NOT_ENOUGH_QUOTA (OMRPORT_ERROR_FILE_BASE-32)
#define OMRPORT_ERROR_FILE_INSUFFICIENT_BUFFER (OMRPORT_ERROR_FILE_BASE-33)
#define OMRPORT_ERROR_FILE_HANDLE_EOF (OMRPORT_ERROR_FILE_BASE-34)
#define OMRPORT_ERROR_FILE_BROKEN_PIPE (OMRPORT_ERROR_FILE_BASE-35)
#define OMRPORT_ERROR_FILE_MORE_DATA (OMRPORT_ERROR_FILE_BASE-36)
#define OMRPORT_ERROR_FILE_INVALID_PARAMETER (OMRPORT_ERROR_FILE_BASE-37)
#define OMRPORT_ERROR_FILE_IO_PENDING (OMRPORT_ERROR_FILE_BASE-38)
#define OMRPORT_ERROR_FILE_READ_NO_BYTES_READ (OMRPORT_ERROR_FILE_BASE-39)
#define OMRPORT_ERROR_FILE_FAILED_TO_ALLOCATE_TLS (OMRPORT_ERROR_FILE_BASE-40)
#define OMRPORT_ERROR_FILE_TOO_MANY_OPEN_FILES (OMRPORT_ERROR_FILE_BASE-41)


/** @} */

/**
 * @name Memory Mapping Errors
 * Error codes for file operations.
 *
 * @internal OMRPORT_ERROR_MMAP* range from -150 to -199 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_MMAP_BASE -150
#define OMRPORT_ERROR_MMAP_MAP_FILE_INVALIDFILE (OMRPORT_ERROR_MMAP_BASE)
#define OMRPORT_ERROR_MMAP_MAP_FILE_UNSUPPORTEDWRITE (OMRPORT_ERROR_MMAP_BASE-1)
#define OMRPORT_ERROR_MMAP_MAP_FILE_INVALIDFLAGS (OMRPORT_ERROR_MMAP_BASE-2)
#define OMRPORT_ERROR_MMAP_MAP_FILE_MALLOCFAILED (OMRPORT_ERROR_MMAP_BASE-3)
#define OMRPORT_ERROR_MMAP_MAP_FILE_CREATEMAPPINGOBJECTFAILED (OMRPORT_ERROR_MMAP_BASE-4)
#define OMRPORT_ERROR_MMAP_MAP_FILE_MAPPINGFAILED (OMRPORT_ERROR_MMAP_BASE-5)
#define OMRPORT_ERROR_MMAP_MSYNC_INVALIDFLAGS (OMRPORT_ERROR_MMAP_BASE-6)
#define OMRPORT_ERROR_MMAP_MSYNC_FAILED (OMRPORT_ERROR_MMAP_BASE-7)
#define OMRPORT_ERROR_MMAP_MAP_FILE_STATFAILED (OMRPORT_ERROR_MMAP_BASE-8)
/** @} */

/**
 * @name memory protection errors
 * Error codes for file operations.
 *
 * @internal OMRPORT_ERROR_MMAP* range from -200 to -249 avoid overlap
 * @{
 */
#define OMRPORT_PAGE_PROTECT_BASE -200
#define OMRPORT_PAGE_PROTECT_FAILED (OMRPORT_PAGE_PROTECT_BASE)
/** @} */

/**
 * @name vmem Errors
 * Error codes for virtual memory operations.
 *
 * @internal OMRPORT_ERROR_VMEM* range from -250 to -299 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_VMEM_BASE -250
#define OMRPORT_ERROR_VMEM_OPFAILED (OMRPORT_ERROR_VMEM_BASE)
#define OMRPORT_ERROR_VMEM_INSUFFICENT_RESOURCES (OMRPORT_ERROR_VMEM_BASE -1)
#define OMRPORT_ERROR_VMEM_INVALID_PARAMS (OMRPORT_ERROR_VMEM_BASE -2)
#define OMRPORT_ERROR_VMEM_NOT_SUPPORTED (OMRPORT_ERROR_VMEM_BASE -3)
#define OMRPORT_ERROR_VMEM_DOUBLE_MAP_FAILED (OMRPORT_ERROR_VMEM_BASE -4)
#define OMRPORT_ERROR_VMEM_DOUBLE_MAP_ADRESS_RESERVE_FAILED (OMRPORT_ERROR_VMEM_BASE -5)

/** @} */

/**
 * @name signal Errors
 * Error codes for signal memory operations.
 *
 * @internal OMRPORT_ERROR_SIG* range from -300 to -349 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_SIG_BASE -300
#define OMRPORT_ERROR_SIG_OPFAILED (OMRPORT_ERROR_VMEM_BASE)
/** @} */

/**
 * @name System V IPC Errors
 * Unix System V IPC function call errno mappings
 * @internal
 * System V errno's for 8 sysv functions:
 * 1 General Function: ftok
 * 3 Semaphore Functions: semget, semctl, semop
 * 4 Memory Functions: shmget, shmctl, shmat, shmdt
 * 
 * The errors are meant to be a 32 bit integer.
 * The most significant 16 bits will identify the failing function 
 * The least significant 16 bits will identify the errno that occured while calling the function
 * 
 * 9 ranges will be used to represent errors starting at the below addresses:
 * 	0xFFFDxxxx for ftok
 * 	0xFFFCxxxx for semget
 * 	0xFFFBxxxx for semctl
 * 	0xFFFAxxxx for semop
 * 	0xFFF9xxxx for shmget
 * 	0xFFF8xxxx for shmctl
 * 	0xFFF7xxxx for shmat
 * 	0xFFF6xxxx for shmdt
 * 	0xFFF5xxxx for __getipc (zOS)
 * @{
 */
#define OMRPORT_ERROR_SYSV_IPC_FTOK_ERROR ((I_32)((U_32)-2 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SEMGET_ERROR ((I_32)((U_32)-3 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SEMCTL_ERROR ((I_32)((U_32)-4 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SEMOP_ERROR ((I_32)((U_32)-5 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SHMGET_ERROR ((I_32)((U_32)-6 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SHMCTL_ERROR ((I_32)((U_32)-7 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SHMAT_ERROR ((I_32)((U_32)-8 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_SHMDT_ERROR ((I_32)((U_32)-9 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))
#define OMRPORT_ERROR_SYSV_IPC_GETIPC_ERROR ((I_32)((U_32)-10 << OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT))

#define OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE -750
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EACCES (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EEXIST (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-1)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ENOENT (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-2)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EINVAL (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-3)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ENOMEM (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-4)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ENOSPC (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-5)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ELOOP (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-6)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ENAMETOOLONG (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-7) 
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ENOTDIR (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-8)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EPERM (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-9)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_ERANGE (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-10)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_E2BIG (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-11)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EAGAIN (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-12)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EFBIG (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-13)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EIDRM (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-14)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EINTR (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-15)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_EMFILE (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-16)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_UNMAPPED (OMRPORT_ERROR_SYSV_IPC_ERRNO_BASE-17)
#define OMRPORT_ERROR_SYSV_IPC_ERRNO_END OMRPORT_ERROR_SYSV_IPC_ERRNO_UNMAPPED
/** @} */

/**
 * @name Shared Semaphore Errors
 * Error codes for shared semaphore operations.
 *
 * @internal OMRPORT_ERROR_SHSEM* range from at -150 to -169.  to match hyporterror.h codes.
 * Overlaps with omrporterror.h
 * @{
 */
#define OMRPORT_ERROR_SHSEM_BASE -150
#define OMRPORT_ERROR_SHSEM_OPFAILED (OMRPORT_ERROR_SHSEM_BASE)
#define OMRPORT_ERROR_SHSEM_HANDLE_INVALID (OMRPORT_ERROR_SHSEM_BASE-1)
#define OMRPORT_ERROR_SHSEM_SEMSET_INVALID (OMRPORT_ERROR_SHSEM_BASE-2)
#define OMRPORT_ERROR_SHSEM_OPFAILED_DONT_UNLINK (OMRPORT_ERROR_SHSEM_BASE-3)
#define OMRPORT_ERROR_SHSEM_OPFAILED_CONTROL_FILE_CORRUPT (OMRPORT_ERROR_SHSEM_BASE-4)
#define OMRPORT_ERROR_SHSEM_OPFAILED_SEMID_MISMATCH (OMRPORT_ERROR_SHSEM_BASE-5)
#define OMRPORT_ERROR_SHSEM_OPFAILED_SEM_KEY_MISMATCH (OMRPORT_ERROR_SHSEM_BASE-6)
#define OMRPORT_ERROR_SHSEM_OPFAILED_SEM_SIZE_CHECK_FAILED (OMRPORT_ERROR_SHSEM_BASE-7)
#define OMRPORT_ERROR_SHSEM_OPFAILED_SEM_MARKER_CHECK_FAILED (OMRPORT_ERROR_SHSEM_BASE-8)
#define OMRPORT_ERROR_SHSEM_DATA_DIRECTORY_FAILED (OMRPORT_ERROR_SHSEM_BASE-9)
#define OMRPORT_ERROR_SHSEM_WAIT_FOR_CREATION_MUTEX_TIMEDOUT (OMRPORT_ERROR_SHSEM_BASE-10)
#define OMRPORT_ERROR_SHSEM_STAT_BUFFER_INVALID (OMRPORT_ERROR_SHSEM_BASE-11)
#define OMRPORT_ERROR_SHSEM_STAT_FAILED (OMRPORT_ERROR_SHSEM_BASE-12)
#define OMRPORT_ERROR_SHSEM_OPFAILED_SEMAPHORE_NOT_FOUND (OMRPORT_ERROR_SHSEM_BASE-13)
#define OMRPORT_ERROR_SHSEM_OPFAILED_CONTROL_FILE_LOCK_FAILED (OMRPORT_ERROR_SHSEM_BASE-14)
 
 /**
 * @name Shared Memory Errors
 * Error codes for shared memory semaphore operations.
 *
 * @internal OMRPORT_ERROR_SHMEM* range from at -170 to -189 to match hyporterror.h codes.
 * Overlaps with omrporterror.h
 * @{
 */
#define OMRPORT_ERROR_SHMEM_BASE -170
#define OMRPORT_ERROR_SHMEM_OPFAILED (OMRPORT_ERROR_SHMEM_BASE)
#define OMRPORT_ERROR_SHMEM_OPEN_ATTACHED_FAILED (OMRPORT_ERROR_SHMEM_BASE-1)
#define OMRPORT_ERROR_SHMEM_CREATE_ATTACHED_FAILED (OMRPORT_ERROR_SHMEM_BASE-2)
#define OMRPORT_ERROR_SHMEM_NOSPACE (OMRPORT_ERROR_SHMEM_BASE-3)
#define OMRPORT_ERROR_SHMEM_TOOBIG (OMRPORT_ERROR_SHMEM_BASE-4)
#define OMRPORT_ERROR_SHMEM_DATA_DIRECTORY_FAILED (OMRPORT_ERROR_SHMEM_BASE-5)
#define OMRPORT_ERROR_SHMEM_WAIT_FOR_CREATION_MUTEX_TIMEDOUT (OMRPORT_ERROR_SHMEM_BASE-6)
#define OMRPORT_ERROR_SHMEM_ZOS_STORAGE_KEY_READONLY (OMRPORT_ERROR_SHMEM_BASE-7)
#define OMRPORT_ERROR_SHMEM_OPFAILED_CONTROL_FILE_CORRUPT (OMRPORT_ERROR_SHMEM_BASE-8)
#define OMRPORT_ERROR_SHMEM_OPFAILED_SHMID_MISMATCH (OMRPORT_ERROR_SHMEM_BASE-9)
#define OMRPORT_ERROR_SHMEM_OPFAILED_SHM_KEY_MISMATCH (OMRPORT_ERROR_SHMEM_BASE-10)
#define OMRPORT_ERROR_SHMEM_OPFAILED_SHM_GROUPID_CHECK_FAILED (OMRPORT_ERROR_SHMEM_BASE-11)
#define OMRPORT_ERROR_SHMEM_OPFAILED_SHM_USERID_CHECK_FAILED (OMRPORT_ERROR_SHMEM_BASE-12)
#define OMRPORT_ERROR_SHMEM_OPFAILED_SHM_SIZE_CHECK_FAILED (OMRPORT_ERROR_SHMEM_BASE-13)
#define OMRPORT_ERROR_SHMEM_HANDLE_INVALID (OMRPORT_ERROR_SHMEM_BASE-14)
#define OMRPORT_ERROR_SHMEM_STAT_BUFFER_INVALID (OMRPORT_ERROR_SHMEM_BASE-15)
#define OMRPORT_ERROR_SHMEM_STAT_FAILED (OMRPORT_ERROR_SHMEM_BASE-16)
#define OMRPORT_ERROR_SHMEM_OPFAILED_SHARED_MEMORY_NOT_FOUND (OMRPORT_ERROR_SHMEM_BASE-17)
#define OMRPORT_ERROR_SHMEM_OPFAILED_CONTROL_FILE_LOCK_FAILED (OMRPORT_ERROR_SHMEM_BASE-18)
/** @} */

#define OMRPORT_ERROR_SYSTEM_CALL_ERRNO_MASK 0xffff0000
#define OMRPORT_ERROR_SYSTEM_CALL_CODE_SHIFT 16

/**
 * Miscellaneous system function call errno mappings
 * @internal
 * Errno for following system call functions:
 * 	getgroups
 *
 * The errors are meant to be a 32 bit integer.
 * The most significant 16 bits will identify the failing function
 * The least significant 16 bits will identify the errno that occured while calling the function
 *
 */
#define OMRPORT_ERROR_SYSINFO_GETGROUPSSIZE_ERROR (-(2<<16))
#define OMRPORT_ERROR_SYSINFO_GETGROUPS_ERROR (-(3<<16))

/**
 * @name sysinfo Errors
 * Error codes for sysinfo operations.
 *
 * @internal OMRPORT_ERROR_SYSINFO* range from -350 to -399 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_SYSINFO_BASE -350
#define OMRPORT_ERROR_SYSINFO_OPFAILED (OMRPORT_ERROR_SYSINFO_BASE)
#define OMRPORT_ERROR_SYSINFO_ENV_INIT_CRASHED_COPYING_BUFFER (OMRPORT_ERROR_SYSINFO_BASE-1)
#define OMRPORT_ERROR_SYSINFO_GET_STATS_FAILED (OMRPORT_ERROR_SYSINFO_BASE-2)
#define OMRPORT_ERROR_SYSINFO_INVALID_TIME (OMRPORT_ERROR_SYSINFO_BASE-3)
#define OMRPORT_ERROR_SYSINFO_QUERY_SETUP_FAILED (OMRPORT_ERROR_SYSINFO_BASE-4)
#define OMRPORT_ERROR_SYSINFO_NOT_SUPPORTED (OMRPORT_ERROR_SYSINFO_BASE-5)
#define OMRPORT_ERROR_SYSINFO_INSUFFICIENT_PRIVILEGE (OMRPORT_ERROR_SYSINFO_BASE-6)
#define OMRPORT_ERROR_SYSINFO_NULL_OBJECT_RECEIVED (OMRPORT_ERROR_SYSINFO_BASE-7)
#define OMRPORT_ERROR_SYSINFO_ERROR_READING_MEMORY_INFO (OMRPORT_ERROR_SYSINFO_BASE-8)
#define OMRPORT_ERROR_SYSINFO_PARAM_HAS_INVALID_RANGE (OMRPORT_ERROR_SYSINFO_BASE-9)
#define OMRPORT_ERROR_SYSINFO_ERROR_READING_PROCESSOR_INFO (OMRPORT_ERROR_SYSINFO_BASE-10)
#define OMRPORT_ERROR_SYSINFO_MEMORY_ALLOC_FAILED (OMRPORT_ERROR_SYSINFO_BASE-11)
#define OMRPORT_ERROR_SYSINFO_UNEXPECTED_PROCESSOR_COUNT (OMRPORT_ERROR_SYSINFO_BASE-12)
#define OMRPORT_ERROR_SYSINFO_ERROR_EINVAL (OMRPORT_ERROR_SYSINFO_BASE-13)
#define OMRPORT_ERROR_SYSINFO_ERROR_EFAULT (OMRPORT_ERROR_SYSINFO_BASE-14)
#define OMRPORT_ERROR_SYSINFO_PROCESSOR_COUNT_UNSTABLE (OMRPORT_ERROR_SYSINFO_BASE-15)
#define OMRPORT_ERROR_SYSINFO_GET_OPEN_FILES_NOT_SUPPORTED (OMRPORT_ERROR_SYSINFO_BASE-16)
#define OMRPORT_ERROR_SYSINFO_CGROUP_UNSUPPORTED_PLATFORM (OMRPORT_ERROR_SYSINFO_BASE-17)
#define OMRPORT_ERROR_SYSINFO_PROCESS_CGROUP_FILE_FOPEN_FAILED (OMRPORT_ERROR_SYSINFO_BASE-18)
#define OMRPORT_ERROR_SYSINFO_PROCESS_CGROUP_FILE_READ_FAILED (OMRPORT_ERROR_SYSINFO_BASE-19)
#define OMRPORT_ERROR_SYSINFO_SYS_FS_CGROUP_STATFS_FAILED (OMRPORT_ERROR_SYSINFO_BASE-20)
#define OMRPORT_ERROR_SYSINFO_SYS_FS_CGROUP_TMPFS_NOT_MOUNTED (OMRPORT_ERROR_SYSINFO_BASE-21)
#define OMRPORT_ERROR_SYSINFO_CGROUP_SUBSYSTEM_UNAVAILABLE (OMRPORT_ERROR_SYSINFO_BASE-22)
#define OMRPORT_ERROR_SYSINFO_CGROUP_NAME_NOT_AVAILABLE (OMRPORT_ERROR_SYSINFO_BASE-23)
#define OMRPORT_ERROR_SYSINFO_CGROUP_SUBSYSTEM_FILE_FOPEN_FAILED (OMRPORT_ERROR_SYSINFO_BASE-24)
#define OMRPORT_ERROR_SYSINFO_CGROUP_MEMLIMIT_NOT_SET (OMRPORT_ERROR_SYSINFO_BASE-25)
#define OMRPORT_ERROR_SYSINFO_CGROUP_SUBSYSTEM_FILE_INVALID_VALUE (OMRPORT_ERROR_SYSINFO_BASE-26)
#define OMRPORT_ERROR_SYSINFO_CGROUP_SUBSYSTEM_METRIC_NOT_AVAILABLE (OMRPORT_ERROR_SYSINFO_BASE-27)
#define OMRPORT_ERROR_SYSINFO_CGROUP_VERSION_NOT_AVAILABLE (OMRPORT_ERROR_SYSINFO_BASE-28)
#define OMRPORT_ERROR_SYSINFO_CGROUP_FILENAME_INVALID (OMRPORT_ERROR_SYSINFO_BASE-29)
#define OMRPORT_ERROR_SYSINFO_CGROUP_NULL_PARAM (OMRPORT_ERROR_SYSINFO_BASE-30)

/**
 * @name Port library initialization return codes
 * Reasons for failure to initialize port library
 *
 * @internal OMRPORT_ERROR_STRING_* range from -400 to -449 avoid overlap
 */

/* default value */
#define OMRPORT_ERROR_INIT_OMR_FAIL_BASE -400
#define OMRPORT_ERROR_INIT_OMR_WRONG_SIZE (OMRPORT_ERROR_INIT_OMR_FAIL_BASE)

/** @} */

/**
 * @name omrstr Errors
 * Error code returned by the string processing API
 *
 * @internal OMRPORT_ERROR_STRING_* range from -450 to -499 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_STRING_BASE -450
#define OMRPORT_ERROR_STRING_ILLEGAL_STRING (OMRPORT_ERROR_STRING_BASE - 0)
#define OMRPORT_ERROR_STRING_BUFFER_TOO_SMALL (OMRPORT_ERROR_STRING_BASE - 1)
#define OMRPORT_ERROR_STRING_ICONV_OPEN_FAILED (OMRPORT_ERROR_STRING_BASE - 2)
#define OMRPORT_ERROR_STRING_UNSUPPORTED_ENCODING (OMRPORT_ERROR_STRING_BASE - 3)
#define OMRPORT_ERROR_STRING_MEM_ALLOCATE_FAILED (OMRPORT_ERROR_STRING_BASE - 4)
/**
 * @}
 */

/**
 * @name OMR Socket API Errors
 * Error code returned by the socket API
 *
 * @internal OMRPORT_ERROR_SOCK_* range from -500 to -549 avoid overlap
 * @{
 */
#define OMRPORT_ERROR_SOCK_BASE -500
#define OMRPORT_ERROR_SOCK_PTB_FAILED (OMRPORT_ERROR_SOCK_BASE - 0)
#define OMRPORT_ERROR_SOCK_ADDRINFO_FAILED (OMRPORT_ERROR_SOCK_BASE - 1)
#define OMRPORT_ERROR_SOCK_SOCKET_CLOSE_FAILED (OMRPORT_ERROR_SOCK_BASE - 2)
#define OMRPORT_ERROR_SOCK_LEVEL_UNSUPPORTED (OMRPORT_ERROR_SOCK_BASE - 3)
#define OMRPORT_ERROR_SOCK_OPTION_UNSUPPORTED (OMRPORT_ERROR_SOCK_BASE - 4)
#define OMRPORT_ERROR_SOCKET_NO_ACCESS (OMRPORT_ERROR_SOCK_BASE - 5)
#define OMRPORT_ERROR_SOCKET_ADDRINUSE (OMRPORT_ERROR_SOCK_BASE - 6)
#define OMRPORT_ERROR_SOCKET_ADDRNOTAVAIL (OMRPORT_ERROR_SOCK_BASE - 7)
#define OMRPORT_ERROR_SOCKET_AF_UNSUPPORTED (OMRPORT_ERROR_SOCK_BASE - 8)
#define OMRPORT_ERROR_SOCKET_WOULDBLOCK (OMRPORT_ERROR_SOCK_BASE - 9)
#define OMRPORT_ERROR_SOCKET_BAD_FILE_DESCRIPTOR (OMRPORT_ERROR_SOCK_BASE - 10)
#define OMRPORT_ERROR_SOCKET_CONNABORTED (OMRPORT_ERROR_SOCK_BASE - 11)
#define OMRPORT_ERROR_SOCKET_CONNREFUSED (OMRPORT_ERROR_SOCK_BASE - 12)
#define OMRPORT_ERROR_SOCKET_CONNRESET (OMRPORT_ERROR_SOCK_BASE - 13)
#define OMRPORT_ERROR_SOCKET_DESTADDRREQ (OMRPORT_ERROR_SOCK_BASE - 14)
#define OMRPORT_ERROR_SOCKET_DOMAIN (OMRPORT_ERROR_SOCK_BASE - 15)
#define OMRPORT_ERROR_SOCKET_BAD_ADDRESS (OMRPORT_ERROR_SOCK_BASE - 16)
#define OMRPORT_ERROR_SOCKET_INPROGRESS (OMRPORT_ERROR_SOCK_BASE - 17)
#define OMRPORT_ERROR_SOCKET_INTERRUPTED (OMRPORT_ERROR_SOCK_BASE - 18)
#define OMRPORT_ERROR_SOCKET_IO (OMRPORT_ERROR_SOCK_BASE - 19)
#define OMRPORT_ERROR_SOCKET_ALREADY_CONNECTED (OMRPORT_ERROR_SOCK_BASE - 20)
#define OMRPORT_ERROR_SOCKET_PATH_NAME_LOOP (OMRPORT_ERROR_SOCK_BASE - 21)
#define OMRPORT_ERROR_SOCKET_MSGSIZE (OMRPORT_ERROR_SOCK_BASE - 22)
#define OMRPORT_ERROR_SOCKET_MAX_FILE_OPENED_PROCESS (OMRPORT_ERROR_SOCK_BASE - 23)
#define OMRPORT_ERROR_SOCKET_NAMETOOLONG (OMRPORT_ERROR_SOCK_BASE - 24)
#define OMRPORT_ERROR_SOCKET_NETDOWN (OMRPORT_ERROR_SOCK_BASE - 25)
#define OMRPORT_ERROR_SOCKET_NETUNREACH (OMRPORT_ERROR_SOCK_BASE - 26)
#define OMRPORT_ERROR_SOCKET_MAX_FILE_OPENED_SYSTEM (OMRPORT_ERROR_SOCK_BASE - 27)
#define OMRPORT_ERROR_SOCKET_NO_BUFFERS (OMRPORT_ERROR_SOCK_BASE - 28)
#define OMRPORT_ERROR_SOCKET_NO_DEVICE (OMRPORT_ERROR_SOCK_BASE - 29)
#define OMRPORT_ERROR_SOCKET_NO_FILE_ENTRY (OMRPORT_ERROR_SOCK_BASE - 30)
#define OMRPORT_ERROR_SOCKET_NOMEM (OMRPORT_ERROR_SOCK_BASE - 31)
#define OMRPORT_ERROR_SOCKET_SYSTEM_FUNCTION_NOT_IMPLEMENTED (OMRPORT_ERROR_SOCK_BASE - 32)
#define OMRPORT_ERROR_SOCKET_NOT_CONNECTED (OMRPORT_ERROR_SOCK_BASE - 33)
#define OMRPORT_ERROR_SOCKET_NOTDIR (OMRPORT_ERROR_SOCK_BASE - 34)
#define OMRPORT_ERROR_SOCKET_NOTSOCK (OMRPORT_ERROR_SOCK_BASE - 35)
#define OMRPORT_ERROR_SOCKET_INVALID_OPERATION (OMRPORT_ERROR_SOCK_BASE - 36)
#define OMRPORT_ERROR_SOCKET_OPERATION_NOT_PERMITTED (OMRPORT_ERROR_SOCK_BASE - 37)
#define OMRPORT_ERROR_SOCKET_BROKEN_PIPE (OMRPORT_ERROR_SOCK_BASE - 38)
#define OMRPORT_ERROR_SOCKET_PROTOCOL_UNSUPPORTED (OMRPORT_ERROR_SOCK_BASE - 39)
#define OMRPORT_ERROR_SOCKET_BAD_PROTOCOL (OMRPORT_ERROR_SOCK_BASE - 40)
#define OMRPORT_ERROR_SOCKET_ROFS (OMRPORT_ERROR_SOCK_BASE - 41)
#define OMRPORT_ERROR_SOCKET_SOCKTYPE_UNSUPPORTED (OMRPORT_ERROR_SOCK_BASE - 42)
#define OMRPORT_ERROR_SOCKET_TIMEOUT (OMRPORT_ERROR_SOCK_BASE - 43)
#define OMRPORT_ERROR_SOCKET_EXCEED_MAX_NFDS (OMRPORT_ERROR_SOCK_BASE - 44)
/**
 * @}
 */

#endif /* omrporterror_h */
