/*******************************************************************************
 * Copyright (c) 2019, 2020 IBM Corp. and others
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] http://openjdk.java.net/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0 WITH Classpath-exception-2.0 OR LicenseRef-GPL-2.0 WITH Assembly-exception
 *******************************************************************************/
package com.ibm.j9ddr.corereaders.elf;

import java.io.IOException;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import com.ibm.j9ddr.corereaders.InvalidDumpFormatException;

/**
 * ELFDumpReader for AArch64 dumps.
 */
public class ELFAArch64DumpReader extends ELFDumpReader {

	ELFAArch64DumpReader(ELFFileReader reader) throws IOException, InvalidDumpFormatException
	{
		super(reader);
	}

	@Override
	protected long readUID() throws IOException {
		return _reader.readInt() & 0xffffffffL;
	}

	@Override
	protected String getProcessorType() {
		return "arm64";
	}

	@Override
	protected SortedMap<String, Number> readRegisters() throws IOException {
		// The first few register names are simple "x"+num
		SortedMap<String, Number> registers = new TreeMap<String, Number>(new RegisterComparator());
		for (int i = 0; i < 30; i++) {
			registers.put( "x"+i, _reader.readLong() );
		}
		// The next registers have names
		registers.put( "lr", _reader.readLong() );
		registers.put( "sp", _reader.readLong() );
		registers.put( "pc", _reader.readLong() );
		registers.put( "pstate", _reader.readLong() );

		return registers;
	}

	@Override
	protected String getStackPointerRegisterName() {
		return "sp";
	}

	@Override
	protected long getBasePointerFrom(Map<String, Number> registers) {
		return getStackPointerFrom(registers);
	}

	@Override
	protected long getInstructionPointerFrom(Map<String, Number> registers) {
		return registers.get("pc").longValue();
	}

	@Override
	protected long getLinkRegisterFrom(Map<String, Number> registers) {
		return registers.get("lr").longValue();
	}

	@Override
	protected void readHighwordRegisters(DataEntry entry,
			Map<String, Number> registers) throws IOException,
			InvalidDumpFormatException {
		throw new InvalidDumpFormatException("Unexpected data entry in AArch64 ELF dump");
}

	@Override
	protected String[] getDwarfRegisterKeys() {
		// Needs to be overridden by each platform. Need to consult the platform ABI to map reg names to dwarf numbers.
		// (This list generated by looking at the order of the registers displayed by info registers in gdb)
		String[] registersVals = new String[34];
		for (int i = 0; i < 30; i++) {
			registersVals[i] = "x"+i;
		}
		registersVals[30] = "lr";
		registersVals[31] = "sp";
		registersVals[32] = "pc";
		registersVals[33] = "sptate";
		return registersVals;
	}

}
