/*******************************************************************************
 * Copyright (c) 1998, 2022 IBM Corp. and others
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] http://openjdk.java.net/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0 WITH Classpath-exception-2.0 OR LicenseRef-GPL-2.0 WITH Assembly-exception
 *******************************************************************************/
#include <string.h>
#include <stdlib.h>
#include "jcl.h"
#include "j9lib.h"
#include "omr.h"
#include "util_api.h"


#include "j9protos.h"
#include "j9consts.h"
#include "omrgcconsts.h"
#include "j2sever.h"

#include "jclglob.h"
#include "jclprots.h"

#include "ut_j9jcl.h"

#if defined(J9ZOS390)
#include "atoe.h"
#include <_Ccsid.h>
#endif



/* JCL_J2SE */
#define JCL_J2SE



typedef struct {
	int errorOccurred;
	jobject args;
	jint nCommandLineDefines;
	JNIEnv *env;
	const char **defaultValues;
	int defaultCount;
	jclass stringClass;
} CreateSystemPropertiesData;

jint propertyListAddString( JNIEnv *env, jarray array, jint arrayIndex, const char *value);
static void JNICALL systemPropertyIterator(char* key, char* value, void* userData);
char* getDefinedEncoding(JNIEnv *env, char *defArg);
jobject getPropertyList(JNIEnv *env);

#if JAVA_SPEC_VERSION >= 11
void JNICALL
Java_java_lang_System_initJCLPlatformEncoding(JNIEnv *env, jclass clazz)
{
	UDATA handle = 0;
	J9JavaVM * const vm = ((J9VMThread*)env)->javaVM;
	char dllPath[EsMaxPath] = {0};
	UDATA written = 0;
	const char *encoding = NULL;
	PORT_ACCESS_FROM_ENV(env);

#if defined(OSX)
	encoding = "UTF-8";
#else
	char property[128] = {0};
	encoding = getPlatformFileEncoding(env, property, sizeof(property), 1); /* platform encoding */
#endif /* defined(OSX) */
	/* libjava.[so|dylib] is in the jdk/lib/ directory, one level up from the default/ & compressedrefs/ directories */
	written = j9str_printf(PORTLIB, dllPath, sizeof(dllPath), "%s/../java", vm->j2seRootDirectory);
	/* Assert the number of characters written (not including the null) fit within the dllPath buffer */
	Assert_JCL_true(written < (sizeof(dllPath) - 1));
	if (0 == j9sl_open_shared_library(dllPath, &handle, J9PORT_SLOPEN_DECORATE)) {
		void (JNICALL *nativeFuncAddrJNU)(JNIEnv *env, const char *str) = NULL;
		if (0 == j9sl_lookup_name(handle, "InitializeEncoding", (UDATA*) &nativeFuncAddrJNU, "VLL")) {
			/* invoke JCL native to initialize platform encoding explicitly */
			nativeFuncAddrJNU(env, encoding);
		}
	}
}
#endif /* JAVA_SPEC_VERSION >= 11 */

/**
 * sysPropID
 *    0 - os.version
 *    1 - platform encoding
 *    2 - file.encoding
 *    3 - os.encoding
 */
jstring JNICALL
Java_java_lang_System_getSysPropBeforePropertiesInitialized(JNIEnv *env, jclass clazz, jint sysPropID)
{
	const char *sysPropValue = NULL;
	/* The sysPropValue points to following property which has to be declared at top level. */
	char property[128] = {0};
	jstring result = NULL;
	PORT_ACCESS_FROM_ENV(env);

	switch (sysPropID) {
	case 0: /* os.version */
		/* Same logic as vmprops.c:initializeSystemProperties(vm) - j9sysinfo_get_OS_version() */
		sysPropValue = j9sysinfo_get_OS_version();
		if (NULL != sysPropValue) {
#if defined(WIN32)
			char *cursor = strchr(sysPropValue, ' ');
			if (NULL != cursor) {
				*cursor = '\0';
			}
#endif /* defined(WIN32) */
		} else {
			sysPropValue = "unknown";
		}
		break;

	case 1: /* platform encoding: ibm.system.encoding, sun.jnu.encoding, native.encoding when file.encoding is NULL */
#if defined(OSX)
		sysPropValue = "UTF-8";
#else
		sysPropValue = getPlatformFileEncoding(env, property, sizeof(property), sysPropID);
#endif /* defined(OSX) */
		break;

	case 2: /* file.encoding */
		sysPropValue = getDefinedEncoding(env, "-Dfile.encoding=");
		if (NULL == sysPropValue) {
#if JAVA_SPEC_VERSION < 18
			sysPropValue = getPlatformFileEncoding(env, property, sizeof(property), sysPropID);
#else /* JAVA_SPEC_VERSION < 18 */
			sysPropValue = "UTF-8";
		} else {
			if (0 == strcmp("COMPAT", sysPropValue)) {
				sysPropValue = getPlatformFileEncoding(env, property, sizeof(property), sysPropID);
			}
#endif /* JAVA_SPEC_VERSION < 18 */
		}
#if defined(J9ZOS390)
		if (__CSNameType(sysPropValue) == _CSTYPE_ASCII) {
			__ccsid_t ccsid;
			ccsid = __toCcsid(sysPropValue);
			atoe_setFileTaggingCcsid(&ccsid);
		}
#endif /* defined(J9ZOS390) */
		break;

	case 3: /* os.encoding */
		sysPropValue = getDefinedEncoding(env, "-Dos.encoding=");
		if (NULL == sysPropValue) {
#if defined(J9ZOS390) || defined(J9ZTPF)
			sysPropValue = "ISO8859_1";
#elif defined(WIN32) /* defined(J9ZOS390) || defined(J9ZTPF) */
			sysPropValue = "UTF-8";
#endif /* defined(J9ZOS390) || defined(J9ZTPF) */
		}
		break;

	default:
		break;
	}
	if (NULL != sysPropValue) {
		result = (*env)->NewStringUTF(env, sysPropValue);
	}

	return result;
}

jobject JNICALL Java_java_lang_System_getPropertyList(JNIEnv *env, jclass clazz)
{
	return getPropertyList(env);
}

jstring JNICALL Java_java_lang_System_mapLibraryName(JNIEnv * env, jclass unusedClass, jstring inName)
{
	PORT_ACCESS_FROM_ENV(env);
	jboolean isCopy = FALSE;
	char *outNameUTF;	
	const char *inNameUTF;
	jstring result;

	if (!inName) {
		jclass aClass;
		aClass = (*env)->FindClass(env, "java/lang/NullPointerException");
		if (0 == aClass) {
			return NULL;
		}
		(*env)->ThrowNew(env, aClass, "");
		return NULL;
	}

	inNameUTF = (const char *) (*env)->GetStringUTFChars(env, inName, &isCopy);
	if (!inNameUTF) {
		return NULL;			/* there will be an exception pending */
	}

	if (!(outNameUTF = jclmem_allocate_memory(env, strlen(inNameUTF) + 20))) /* allow for platform extras plus trailing zero */
		return NULL;
	mapLibraryToPlatformName(inNameUTF, outNameUTF);

	(*env)->ReleaseStringUTFChars(env, inName, inNameUTF);

	result = (*env)->NewStringUTF(env, outNameUTF);
	jclmem_free_memory(env, outNameUTF);
	return result;
}


void JNICALL Java_java_lang_System_setFieldImpl(JNIEnv * env, jclass cls, jstring name, jobject stream)
{
	jfieldID descriptorFID;
	const char *bytes;

	bytes = (const char *) (*env)->GetStringUTFChars(env, name, NULL);
	if (!bytes) return;
	if (!strcmp(bytes, "in"))
		descriptorFID = (*env)->GetStaticFieldID(env, cls, bytes, "Ljava/io/InputStream;");
	else
		descriptorFID = (*env)->GetStaticFieldID(env, cls, bytes, "Ljava/io/PrintStream;");
	(*env)->ReleaseStringUTFChars(env, name, bytes);
	if (!descriptorFID) return;
	(*env)->SetStaticObjectField(env, cls, descriptorFID, stream);
}


jobject createSystemPropertyList(JNIEnv *env, const char *defaultValues[], int defaultCount)
{
	VMI_ACCESS_FROM_ENV(env);

	jint i, nCommandLineDefines = 0;
	jclass stringClass;
	jarray args = NULL;
	int propertyCount;

	stringClass = (*env)->FindClass(env, "java/lang/String");
	if (!stringClass) {
/*		printf("\nFailed to find class java/lang/String");*/
		return (jobject) 0;
	}

	(*VMI)->CountSystemProperties(VMI, &propertyCount);
	if (propertyCount) {
		CreateSystemPropertiesData iteratorData;

		args = (*env)->NewObjectArray(env, defaultCount + (propertyCount * 2), stringClass, NULL);
		if (NULL == args) {
			return NULL;
		}

		iteratorData.errorOccurred = 0;
		iteratorData.args = args;
		iteratorData.nCommandLineDefines = nCommandLineDefines;
		iteratorData.env = env;
		iteratorData.defaultValues = defaultValues;
		iteratorData.defaultCount = defaultCount;

		iteratorData.stringClass = stringClass;
		(*VMI)->IterateSystemProperties(VMI, systemPropertyIterator, &iteratorData);
		if (iteratorData.errorOccurred) {
			return NULL;
		}
		nCommandLineDefines = iteratorData.nCommandLineDefines;
	}

	if (NULL == args) {
		args = (*env)->NewObjectArray(env, defaultCount, stringClass, NULL);
	}
	if (NULL == args) {
/*		printf("\nFailed to create arg array");*/
		return NULL;
	}

	for (i = 0; i < defaultCount; ++i) {
		if (defaultValues[i] == NULL) continue;
		if (-1 == propertyListAddString( env, args, nCommandLineDefines, defaultValues[i]) ) {
			return NULL;
		}
		nCommandLineDefines++;
	}

	return args;
}


char* getDefinedEncoding(JNIEnv *env, char *defArg)
{
	VMI_ACCESS_FROM_ENV(env);

	JavaVMInitArgs  *vmInitArgs = (*VMI)->GetInitArgs(VMI);
	int len = (int)strlen(defArg);

	if (vmInitArgs) {
		jint optionIndex;
		JavaVMOption *option = vmInitArgs->options;

		for (optionIndex=0; optionIndex < vmInitArgs->nOptions; optionIndex++) {
			char *optionValue = option->optionString;
			if (strncmp(defArg, optionValue, len) == 0)
				return &optionValue[len];
			option++;
		}
	}
	return NULL;
}

/**
 * @return 0 on success, -1 on error
 */
jint propertyListAddString( JNIEnv *env, jarray array, jint arrayIndex, const char *value)
{
	/* String must be well-formed modified UTF-8 */
	jobject str = (*env)->NewStringUTF(env, value);
	if (NULL != str) {
		(*env)->SetObjectArrayElement(env, array, arrayIndex, str);
	}
	/* NewStringUTF does not throw an exception, other than OutOfMemory */
	return ((*env)->ExceptionCheck(env) == JNI_TRUE) ? -1 : 0;
}

jobject getPropertyList(JNIEnv *env)
{
	PORT_ACCESS_FROM_ENV(env);
	int propIndex = 0;
	jobject propertyList = NULL;
#define PROPERTY_COUNT 137
	char *propertyKey = NULL;
	const char * language = NULL;
	const char * region = NULL;
	const char * variant = NULL;
	const char *strings[PROPERTY_COUNT];
#define USERNAME_LENGTH 128
	char username[USERNAME_LENGTH];
	char *usernameAlloc = NULL;
	/* buffer to hold the size of the maximum direct byte buffer allocations */
	char maxDirectMemBuff[24];
	IDATA result = 0;

	J9JavaVM *javaVM = ((J9VMThread *) env)->javaVM;
	OMR_VM *omrVM = javaVM->omrVM;

	/* Change the allocation value PROPERTY_COUNT above as you add/remove properties,
	 * then follow the propIndex++ convention and consume 2 * slots for each property. 2 * number of property keys is the
	 * correct allocation.
	 * Also note the call to addSystemProperties below, which may add some configuration-specific properties.  Be sure to leave
	 * enough room in the property list for all possibilities.
	 */

	if (J9_GC_POLICY_METRONOME == (omrVM->gcPolicy)) {
		strings[propIndex++] = "com.ibm.jvm.realtime";
		strings[propIndex++] = "soft";
	}

#if defined(J9VM_OPT_SHARED_CLASSES)
	strings[propIndex++] = "com.ibm.oti.shared.enabled";
	if ((((J9VMThread *) env)->javaVM->sharedClassConfig != NULL)
		&& (J9_ARE_ALL_BITS_SET(((J9VMThread *) env)->javaVM->sharedClassConfig->runtimeFlags, J9SHR_RUNTIMEFLAG_ENABLE_CACHE_NON_BOOT_CLASSES))
	) {
		strings[propIndex++] = "true";
	} else {
		strings[propIndex++] = "false";
	}
#endif

#if defined(JCL_J2SE)
	strings[propIndex++] = "ibm.signalhandling.sigchain";
	if (javaVM->sigFlags & J9_SIG_NO_SIG_CHAIN) {
		strings[propIndex++] = "false";
	} else {
		strings[propIndex++] = "true";
	}
	strings[propIndex++] = "ibm.signalhandling.sigint";
	if (javaVM->sigFlags & J9_SIG_NO_SIG_INT) {
		strings[propIndex++] = "false";
	} else {
		strings[propIndex++] = "true";
	}

	/* The JCLs use ibm.signalhandling.rs to determine if they should prevent the registration of signal handlers for what 
	 * 	we consider to be asynchronous signals.
	 * The JCLs do not install handlers for any synchronous signals */
	strings[propIndex++] = "ibm.signalhandling.rs";
	if (J9_ARE_ALL_BITS_SET(javaVM->sigFlags, J9_SIG_XRS_ASYNC)) {
		strings[propIndex++] = "true";
	} else {
		strings[propIndex++] = "false";
	}
#endif

	strings[propIndex++] = "com.ibm.vm.bitmode";
#ifdef J9VM_ENV_DATA64
	strings[propIndex++] = "64";
#else
	strings[propIndex++] = "32";
#endif

	strings[propIndex++] = "com.ibm.cpu.endian";
#ifdef J9VM_ENV_LITTLE_ENDIAN
	strings[propIndex++] = "little";
#else
	strings[propIndex++] = "big";
#endif

	strings[propIndex++] = "sun.cpu.endian";
#ifdef J9VM_ENV_LITTLE_ENDIAN
	strings[propIndex++] = "little";
#else
	strings[propIndex++] = "big";
#endif

/*	Don't set this property as the class library will look here first and when
	there is a security manager you will get a security exception. The code
	looks in this package by default, see URLConnection.getContentHandler()
	strings[propIndex++] = "java.content.handler.pkgs";
	strings[propIndex++] = "com.ibm.oti.www.content";
*/

	/*[PR 95709]*/

	/* Get the language, region and variant */
	language = j9nls_get_language();	
	region = j9nls_get_region();
	variant = j9nls_get_variant();
	
	/* CMVC 144405 : Norwegian Bokmal and Nynorsk need special consideration */
	if ( (strcmp(language, "nn")== 0) && (strcmp(region, "NO") == 0) ){
		variant = "NY";
	}
	if ( (strcmp(language, "nn") == 0) || (strcmp(language, "nb") == 0) ) {
		language = "no";
	}
	
	strings[propIndex++] = "user.language";
	strings[propIndex++] = language;	

	propertyKey = "user.country";
	strings[propIndex++] = propertyKey;
	strings[propIndex++] = region;

	/* Get the variant */
	strings[propIndex++] = "user.variant";
	strings[propIndex++] = variant;

	/* Get the User name */
	strings[propIndex++] = "user.name";
	result = j9sysinfo_get_username(username, USERNAME_LENGTH);
	if (!result) {
		strings[propIndex++] = username;
	} else {
		if (result > 0) {
			usernameAlloc = jclmem_allocate_memory(env, result);
			if (usernameAlloc) {
				result = j9sysinfo_get_username(usernameAlloc, result);
			}
		}
		strings[propIndex++] = !usernameAlloc || result ? "unknown" : usernameAlloc;
	}

#undef USERNAME_LENGTH

#if defined(OPENJ9_BUILD) && JAVA_SPEC_VERSION == 8
	/* Set the maximum direct byte buffer allocation property if it has not been set manually */
	if ((UDATA) -1 == javaVM->directByteBufferMemoryMax) {
		UDATA heapSize = javaVM->memoryManagerFunctions->j9gc_get_maximum_heap_size(javaVM);
		/* allow up to 7/8 of the heap to be direct byte buffers */
		javaVM->directByteBufferMemoryMax = heapSize - (heapSize / 8);
	}
#endif /* defined(OPENJ9_BUILD) && JAVA_SPEC_VERSION == 8 */
#if !defined(OPENJ9_BUILD)
	/* Don't set a default value for IBM Java 8. */
	if ((UDATA) -1 != javaVM->directByteBufferMemoryMax)
#endif /* !defined(OPENJ9_BUILD) */
	{
		strings[propIndex] = "sun.nio.MaxDirectMemorySize";
		propIndex += 1;
		if ((UDATA) -1 == javaVM->directByteBufferMemoryMax) {
			strcpy(maxDirectMemBuff, "-1");
		} else {
			j9str_printf(PORTLIB, maxDirectMemBuff, sizeof(maxDirectMemBuff), "%zu", javaVM->directByteBufferMemoryMax);
		}
		strings[propIndex] = maxDirectMemBuff;
		propIndex += 1;
	}

	propertyList = getPlatformPropertyList(env, strings, propIndex);
	if (NULL != usernameAlloc) {
		jclmem_free_memory(env, usernameAlloc);
	}
	return propertyList;
}

static void JNICALL
systemPropertyIterator(char* key, char* value, void* userData)
{
	CreateSystemPropertiesData * iteratorData = userData;
	jobject args = iteratorData->args;
	JNIEnv *env = iteratorData->env;
	const char **defaultValues = iteratorData->defaultValues;
	int defaultCount = iteratorData->defaultCount;
	jint i;

	/* CMVC 95717: if an error has already occurred get out of here */
	if ( iteratorData->errorOccurred ) {
		return;
	}
	
	if (0 == strcmp("com.ibm.oti.shared.enabled", key)) {
		/* JAZZ103 85641: Prevent com.ibm.oti.shared.enabled from being overwritten by a command line option */
		return;
	}
	

	/* check for overridden system properties, use linear scan for now */
	for (i=0; i < defaultCount; i+=2) {
		if (defaultValues[i] && !strcmp(key, defaultValues[i])) {
			defaultValues[i] = NULL;
			defaultValues[i+1] = NULL;
			break;
		} 
	}

	/* First do the key */
	if( propertyListAddString( env, args, iteratorData->nCommandLineDefines++, key) ) {
		iteratorData->errorOccurred = 1;
		return;
	}

	/* Then the value */
	if( propertyListAddString( env, args, iteratorData->nCommandLineDefines++, value) ) {
		iteratorData->errorOccurred = 1;
		return;
	}

	Trc_JCL_systemPropertyIterator(env, key, value);
}


void JNICALL
Java_java_lang_System_startSNMPAgent(JNIEnv *env, jclass jlClass)
{
	J9VMThread * currentThread = (J9VMThread*)env;
	J9JavaVM * vm = currentThread->javaVM;

	if (J9_ARE_ALL_BITS_SET(vm->jclFlags, J9_JCL_FLAG_COM_SUN_MANAGEMENT_PROP)) {
		jclass smAClass = NULL;
		jmethodID startAgent = NULL;
		
		if (J2SE_VERSION(vm) >= J2SE_V11) {
			smAClass = (*env)->FindClass(env, "jdk/internal/agent/Agent");
		} else {
			smAClass = (*env)->FindClass(env, "sun/management/Agent");
		}
		if (NULL != smAClass) {
			startAgent = (*env)->GetStaticMethodID(env, smAClass, "startAgent", "()V");
			if (NULL != startAgent) {
				(*env)->CallStaticVoidMethod(env, smAClass, startAgent);
			}
		} else {
			(*env)->ExceptionClear(env);
		}
	}
}

void JNICALL
Java_java_lang_System_rasInitializeVersion(JNIEnv * env, jclass unusedClass, jstring javaRuntimeVersion)
{
	J9JavaVM *javaVM = ((J9VMThread *)env)->javaVM;
	jboolean isCopy = JNI_FALSE;
	const char *utfRuntimeVersion = NULL;

	if (NULL != javaRuntimeVersion) {
		utfRuntimeVersion = (*env)->GetStringUTFChars(env, javaRuntimeVersion, &isCopy);
	}

	javaVM->internalVMFunctions->rasSetServiceLevel(javaVM, utfRuntimeVersion);

	if (NULL != utfRuntimeVersion) {
		(*env)->ReleaseStringUTFChars(env, javaRuntimeVersion, utfRuntimeVersion);
	}
}
