#!/bin/sh

# Minimal shell-based init system for Debian and derivatives
#
# It is not meant to be used in production but rather to analyse
# minimal userspace requirements without all the behind-the-scene "magic"
# of a proper init system.
#
# Copyright 2024 Argeo GmbH
# Copyright 2024 Mathieu Baudier
# SPDX-Identifier: GPL-2.0-or-later

SHUTDOWN_FIFO=/run/shutdown.fifo
SHUTDOWN_TIMEOUT=5
echo 1 > /proc/sys/kernel/sysrq # ensure sysrq is enabled

shutdown() {
	level=$1
	echo e > /proc/sysrq-trigger # SIGTERM to all
	sleep $SHUTDOWN_TIMEOUT
	echo i > /proc/sysrq-trigger # SIGKILL to all
	/bin/sync
	echo s > /proc/sysrq-trigger # sync fs
	echo u > /proc/sysrq-trigger # unmount
	sleep 1
	if [ "$level" = "6" ]; then
		echo b > /proc/sysrq-trigger # reboot
	else
		echo o > /proc/sysrq-trigger # shutdown
	fi
	sleep infinity	
}

if [ "$1" = "0" -o "$1" = "6" ]; then # external shutdown
	if [ -e $SHUTDOWN_FIFO ]; then # orderly
		echo "$1" > $SHUTDOWN_FIFO
		exit 0
	else
		shutdown "$1"
	fi
elif [ "$1" = "1" ]; then # single user
	/bin/su - # root session
	shutdown 6 # reboot
fi

## REGULAR INITIALIZATION
set -v # enable shell verbosity

mount -o remount,rw / # mount root fs read/write
[ -e /etc/hostname ] && hostname -F /etc/hostname # set hostname
/etc/init.d/procps start # apply sysctl

# hardeware detection
if [ -e /etc/init.d/udev ]; then
/etc/init.d/udev start
elif [ -e /usr/local/etc/init.d/udev ]; then
/usr/local/etc/init.d/udev start
fi

mount -a # mount all filesystems

[ -e /etc/init.d/qemu-guest-agent ] && /etc/init.d/qemu-guest-agent start # libvirt guest agent
[ -e /etc/init.d/dbus ] && /etc/init.d/dbus start

# start networking asynchronously
[ -e /etc/init.d/networking ] && /etc/init.d/networking start &
[ -e /etc/init.d/ssh ] && /etc/init.d/ssh start &
[ -e /etc/init.d/chrony ] && /etc/init.d/chrony start &

## USER INTERFACE
[ -e /etc/init.d/console-setup.sh ] && /etc/init.d/console-setup.sh start
[ -e /etc/init.d/keyboard-setup.sh ] && /etc/init.d/keyboard-setup.sh start
[ -e /etc/init.d/x11-common ] && /etc/init.d/x11-common start

set +v # disable shell verbosity
if [ -e /usr/sbin/getty ]; then
	/usr/sbin/getty 38400 tty2 &
	/usr/sbin/getty 38400 tty3 &
	/usr/sbin/getty 38400 tty4 &
fi
	
if [ "$1" = "5" -a -e /usr/bin/startx ]; then # graphical
	/usr/bin/startx &
else
	if [ ! -e /usr/sbin/getty ]; then
		echo "Login: "
		read -r INPUT
		/bin/setpriv --reuid=nobody su - $INPUT
		LEVEL=6 # reboot by default
	fi
fi

## WAIT FOR SHUTDOWN
if [ -z "$LEVEL" ]; then
	mkfifo $SHUTDOWN_FIFO # create shutdown trigger
	LEVEL=$(cat $SHUTDOWN_FIFO) # wait
fi
echo "$0 - Initiating shutdown with level $LEVEL..."
shutdown "$LEVEL"
