/*
 * Copyright IBM Corp. and others 2017
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package org.openj9.test.java.lang.management.ThreadMXBean;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;

import org.testng.Assert;
import org.testng.annotations.Optional;
import org.testng.annotations.Parameters;
import org.testng.annotations.Test;

/* getThreadInfo() on a thread that is exited, but not joined. */
public class ExitedThread extends ThreadMXBeanTestCase {
	@Parameters( {"timeoutPara"} )
	@Test(groups = { "level.extended" })
	public void testExitedThread(@Optional("300") String timeoutPara) {
		int timeoutSec = Integer.parseInt(timeoutPara);

		if (timeoutSec < 1)
			timeoutSec = 1;

		InspectorThread inspector = new InspectorThread();
		try {
			inspector.join(timeoutSec * 1000);
			if (inspector.isAlive()) {
				inspector.interrupt();
				inspector.join();
			}
		} catch (InterruptedException e) {
			setExitStatus(ExitStatus.INTERRUPTED);
		}
		Assert.assertEquals(getExitStatus(), ExitStatus.PASS);
	}

	class ExitThread extends Thread {
		Object sync;

		ExitThread(Object sync) {
			this.sync = sync;
		}

		public void run() {
			synchronized (sync) {
				sync.notify();
			}
			/* exit */
		}
	}

	class InspectorThread extends Thread {
		Thread target;

		InspectorThread() {
			this.target = new ExitThread(this);
		}

		public void run() {
			ThreadMXBean mxb = ManagementFactory.getThreadMXBean();
			ThreadInfo info = null;
			boolean done = false;

			try {
				/* wait for the target thread to start */
				synchronized (this) {
					target.start();
					this.wait();
				}
			} catch (InterruptedException e) {
				setExitStatus(ExitStatus.INTERRUPTED);
				return;
			}

			try {
				/* loop until getThreadInfo() returns null */
				while (!done) {
					info = mxb.getThreadInfo(this.target.getId());
					if (info == null) {
						done = true;
					} else if (this.isInterrupted()) {
						done = true;
					}
				}

				/* fail if we timed out before the target exited */
				if (info != null) {
					setFailed();
				}
				target.join();
				
			} catch (InterruptedException e) {
				setExitStatus(ExitStatus.FAIL);
			}
		}
	}
}
